<!-- =========================================================================
 Copyright (c) 2017 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:saxon="http://icl.com/saxon"
  xmlns:exsl="http://exslt.org/common"
  xmlns:p="java:com.xmlmind.w2x.processor.Processor"
  extension-element-prefixes="saxon exsl"
  exclude-result-prefixes="saxon exsl p">

<xsl:param name="section-depth" select="1" />

<xsl:param name="output-path" select="''" />

<xsl:param name="topic-path" select="''" />

<xsl:param name="add-toc" select="'yes'" />

<xsl:param name="add-index" select="'yes'" />

<!-- Not really needed,  except to make a difference between strict task and
     general task. -->
<xsl:param name="topic-type" select="''" />

<xsl:param name="mapType" select="'map'" />

<xsl:output method="xml" encoding="UTF-8" indent="no"
  doctype-public="-//OASIS//DTD DITA Map//EN"
  doctype-system="map.dtd" />

<!-- Generate bookmap or map ============================================= -->

<xsl:template match="/*">
  <xsl:variable name="topicName" select="local-name()" />

  <xsl:variable name="mapName">
    <xsl:choose>
      <xsl:when test="$mapType = 'bookmap'">bookmap</xsl:when>
      <xsl:otherwise>map</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="metaName">
    <xsl:choose>
      <xsl:when test="$mapType = 'bookmap'">bookmeta</xsl:when>
      <xsl:otherwise>topicmeta</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicrefName">
    <xsl:choose>
      <xsl:when test="$mapType = 'bookmap'">chapter</xsl:when>
      <xsl:otherwise>topicref</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:element name="{$mapName}">
    <xsl:copy-of select="@xml:lang" />

    <xsl:copy-of select="./title" />

    <xsl:if test="./shortdesc or ./prolog">
      <xsl:element name="{$metaName}">
        <xsl:copy-of select="./shortdesc" />

        <xsl:copy-of select="./prolog/author" />

        <xsl:if test="./prolog/publisher">
          <xsl:choose>
            <xsl:when test="$mapType = 'bookmap'">
              <publisherinformation>
                <organization>
                  <xsl:copy-of select="./prolog/publisher/node()" />
                </organization>
              </publisherinformation>
            </xsl:when>
            <xsl:otherwise>
              <xsl:copy-of select="./prolog/publisher" />
            </xsl:otherwise>
          </xsl:choose>
        </xsl:if>

        <xsl:copy-of select="./prolog/critdates" />
      </xsl:element>
    </xsl:if>

    <!-- TOC -->

    <xsl:if test="$mapType = 'bookmap' and $add-toc = 'yes'">
      <frontmatter>
        <booklists>
          <toc />
        </booklists>
      </frontmatter>
    </xsl:if>

    <!-- Topmost topic -->

    <xsl:variable name="topicContents"
                  select="./*[local-name() != 'title' and 
                              local-name() != 'shortdesc' and 
                              local-name() != 'prolog' and 
                              local-name() != $topicName]" />
    <xsl:if test="count($topicContents//p) != 1 or 
                  count($topicContents//p//node()) &gt; 0">

      <!-- No need to generate first topic if it has just an empty body. 
           Note that for ensuring the validity of the DITA topic,
           This body may contain an empty p, itself possibly contained in
           a section/prereq/steps-informal wrapper. 
           Therefore we need to detect this specific case. -->

      <xsl:variable name="topicFile">
        <xsl:call-template name="topicFilename">
          <xsl:with-param name="id" select="@id" />
        </xsl:call-template>
      </xsl:variable>

      <xsl:element name="{$topicrefName}">
        <xsl:attribute name="href">
          <xsl:value-of select="$topicFile" />
        </xsl:attribute>
      </xsl:element>

      <xsl:variable name="topic">
        <xsl:element name="{$topicName}">
          <xsl:copy-of select="@*" /> <!-- Including id and xml:lang. -->

          <title><!-- Empty. Already used for the map. --></title>

          <xsl:apply-templates select="$topicContents" mode="copy"/>
        </xsl:element>
      </xsl:variable>

      <xsl:call-template name="writeTopic">
        <xsl:with-param name="file" select="$topicFile" />
        <xsl:with-param name="topic" select="$topic" />
      </xsl:call-template>
    </xsl:if>

    <!-- Nested topics -->

    <xsl:apply-templates select="./*[local-name() = $topicName]" />

    <!-- Index -->

    <xsl:if test="$mapType = 'bookmap' and $add-index = 'yes' and
                  //indexterm">
      <backmatter>
        <booklists>
          <indexlist />
        </booklists>
      </backmatter>
    </xsl:if>
  </xsl:element>
</xsl:template>

<xsl:template name="topicFilename">
  <xsl:param name="id" />

  <xsl:variable name="topicDir">
    <xsl:call-template name="appendSlash">
      <xsl:with-param name="path" select="$topic-path" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:value-of select="concat($topicDir, $id, '.dita')" />
</xsl:template>

<xsl:template name="appendSlash">
  <xsl:param name="path" />

  <xsl:if test="$path != ''">
    <xsl:choose>
      <xsl:when test="substring($path, string-length($path)) = '/'">
        <xsl:value-of select="$path" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat($path, '/')" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:if>
</xsl:template>

<xsl:template match="concept|task|reference|topic">
  <xsl:variable name="topicName" select="local-name()" />

  <xsl:variable name="depth"
                select="count(ancestor::*[local-name() = $topicName])" />

  <xsl:if test="$depth &lt;= $section-depth">
    <!-- .dita file -->

    <xsl:variable name="topic">
      <xsl:element name="{$topicName}">
        <xsl:copy-of select="/*/@xml:lang" /> <!-- Topmost xml:lang -->
        <xsl:copy-of select="@*" /> <!-- May override topmost xml:lang -->

        <xsl:choose>
          <xsl:when test="$depth = $section-depth">
            <!-- Copy all child elements including nested topics. -->
            <xsl:apply-templates select="./*" mode="copy"/>
          </xsl:when>
          <xsl:otherwise>
            <!-- Copy all child elements except nested topics. -->
            <xsl:apply-templates select="./*[local-name() != $topicName]"
                                 mode="copy"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:element>
    </xsl:variable>

    <xsl:variable name="topicFile">
      <xsl:call-template name="topicFilename">
        <xsl:with-param name="id" select="@id" />
      </xsl:call-template>
    </xsl:variable>

    <xsl:call-template name="writeTopic">
      <xsl:with-param name="file" select="$topicFile" />
      <xsl:with-param name="topic" select="$topic" />
    </xsl:call-template>

    <!-- topicref -->

    <xsl:variable name="topicrefName">
      <xsl:choose>
        <xsl:when test="$mapType = 'bookmap' and $depth = 1">chapter</xsl:when>
        <xsl:otherwise>topicref</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:element name="{$topicrefName}">
      <xsl:attribute name="href">
        <xsl:value-of select="$topicFile" />
      </xsl:attribute>

      <xsl:apply-templates select="./*[local-name() = $topicName]" />
    </xsl:element>
  </xsl:if>
</xsl:template>

<!-- Generate topics ===================================================== -->

<!-- image ========================= -->

<xsl:template match="image[@href and 
                           not(starts-with(@href, 'file:/') or 
                               contains(@href, '://'))]"
              mode="copy">
  <xsl:choose>
    <xsl:when test="$topic-path != ''">
      <xsl:variable name="topicDir">
        <xsl:call-template name="appendSlash">
          <xsl:with-param name="path" select="$topic-path" />
        </xsl:call-template>
      </xsl:variable>

      <xsl:variable name="mapDir">
        <xsl:call-template name="topicToMapPath">
          <xsl:with-param name="path" select="$topicDir" />
        </xsl:call-template>
      </xsl:variable>

      <image href="{concat($mapDir, @href)}">
        <xsl:apply-templates select="@*[local-name() != 'href']|node()" 
                             mode="copy" />
      </image>
    </xsl:when>

    <xsl:otherwise>
      <xsl:copy>
        <xsl:apply-templates select="@*|node()" mode="copy" />
      </xsl:copy>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="topicToMapPath">
  <xsl:param name="path" />

  <xsl:if test="contains($path, '/')">
    <xsl:variable name="tail">
      <xsl:call-template name="topicToMapPath">
        <xsl:with-param name="path" select="substring-after($path, '/')" />
      </xsl:call-template>
    </xsl:variable>

    <xsl:value-of select="concat('../', $tail)" />
  </xsl:if>
</xsl:template>

<!-- xref ========================== -->

<xsl:template match="xref[starts-with(@href, '#')]" mode="copy">
  <xsl:variable name="containerTopicId">
    <xsl:call-template name="containerTopicId">
      <xsl:with-param name="contained" select="." />
    </xsl:call-template>
  </xsl:variable>

  <xsl:variable name="href" select="substring-after(@href, '#')" />

  <xsl:variable name="targetTopicId">
    <xsl:choose>
      <xsl:when test="contains($href, '/')">
        <xsl:value-of select="substring-before($href, '/')" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$href" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="targetTopic" select="(//*[@id = $targetTopicId])[1]" />

  <xsl:variable name="containerTopicId2">
    <xsl:if test="$targetTopic">
      <xsl:call-template name="containerTopicId">
        <xsl:with-param name="contained" select="$targetTopic" />
      </xsl:call-template>
    </xsl:if>
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="$containerTopicId != '' and
                    $containerTopicId2 != '' and 
                    $containerTopicId != $containerTopicId2">
      <!-- All topics are always created in the same directory. -->
      <xref href="{concat($containerTopicId2, '.dita#', $href)}">
        <xsl:apply-templates select="@*[local-name() != 'href']|node()" 
                             mode="copy" />
      </xref>
    </xsl:when>

    <xsl:otherwise>
      <xsl:copy>
        <xsl:apply-templates select="@*|node()" mode="copy" />
      </xsl:copy>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="containerTopicId">
  <xsl:param name="contained" />

  <xsl:variable name="topicName" select="local-name(/*)" />

  <xsl:variable name="topic" 
    select="($contained/ancestor-or-self::*[local-name() = $topicName])[last()]" />

  <xsl:if test="$topic">
    <xsl:variable name="containerTopic" 
      select="($topic/ancestor-or-self::*[count(ancestor::*[local-name() = $topicName]) &lt;= $section-depth])[last()]" />

    <xsl:if test="$containerTopic">
      <xsl:value-of select="$containerTopic/@id" />
    </xsl:if>
  </xsl:if>
</xsl:template>

<xsl:template match="@*|node()" mode="copy">
  <xsl:copy>
    <xsl:apply-templates select="@*|node()" mode="copy" />
  </xsl:copy>
</xsl:template>

<xsl:template name="writeTopic">
  <xsl:param name="file" />
  <xsl:param name="topic" />

  <xsl:if test="not(function-available('exsl:node-set'))">
    <xsl:message terminate="yes">
      <xsl:text>Function 'exsl:node-set' not implemented by </xsl:text>
      <xsl:value-of select="system-property('xsl:vendor')"/>
      <xsl:text>.</xsl:text>
    </xsl:message>
  </xsl:if>

  <xsl:variable name="savedTopic" select="exsl:node-set($topic)/*" />

  <!--
  <xsl:message>
    <xsl:value-of select="concat('topic-type=', $topic-type)" />
  </xsl:message>
  -->

  <xsl:variable name="topicType">
    <xsl:choose>
      <xsl:when test="$topic-type != ''">
        <xsl:value-of select="$topic-type" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="local-name($savedTopic)" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicDoctypePublic">
    <xsl:choose>
      <xsl:when
        test="$topicType = 'concept'">-//OASIS//DTD DITA Concept//EN</xsl:when>
      <xsl:when
        test="$topicType = 'generalTask'">-//OASIS//DTD DITA General Task//EN</xsl:when>
      <xsl:when
        test="$topicType = 'task' or
              $topicType = 'strictTask'">-//OASIS//DTD DITA Task//EN</xsl:when>
      <xsl:when
        test="$topicType = 'reference'">-//OASIS//DTD DITA Reference//EN</xsl:when>
      <xsl:otherwise>-//OASIS//DTD DITA Topic//EN</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="topicDoctypeSystem">
    <xsl:choose>
      <xsl:when test="$topicType = 'concept'">concept.dtd</xsl:when>
      <xsl:when test="$topicType = 'generalTask'">generalTask.dtd</xsl:when>
      <xsl:when test="$topicType = 'task' or
                      $topicType = 'strictTask'">task.dtd</xsl:when>
      <xsl:when test="$topicType = 'reference'">reference.dtd</xsl:when>
      <xsl:otherwise>topic.dtd</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:variable name="outputDir">
    <xsl:call-template name="appendSlash">
      <xsl:with-param name="path" select="$output-path" />
    </xsl:call-template>
  </xsl:variable>

  <xsl:variable name="href">
    <xsl:value-of select="concat($outputDir, $file)" />
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="element-available('exsl:document')">
      <exsl:document href="{$href}" method="xml" encoding="UTF-8" indent="no" 
                     doctype-public="{$topicDoctypePublic}" 
                     doctype-system="{$topicDoctypeSystem}">
        <xsl:copy-of select="$savedTopic"/>
      </exsl:document>
    </xsl:when>

    <xsl:when test="element-available('saxon:output')">
      <!-- Saxon 6.5 interprets the href attribute of <xsl:document> as a
           filename relative to the current working directory. You can use an
           absolute filename if you want, but not an absolute URI. This is a
           documented restriction, not a bug. -->

      <xsl:variable name="href2">
        <xsl:choose>
          <xsl:when test="starts-with($href, 'file:/')">
            <xsl:value-of select="p:fileURLToPath($href)" />
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="$href" />
          </xsl:otherwise>
        </xsl:choose>
      </xsl:variable>

      <!--
      <xsl:message>
        <xsl:value-of select="concat('saxon:output href=', $href, 
                                     ' href2=', $href2)" />
      </xsl:message>
      -->

      <saxon:output href="{$href2}" method="xml" encoding="UTF-8" indent="no"
                    doctype-public="{$topicDoctypePublic}"
                    doctype-system="{$topicDoctypeSystem}">
        <xsl:copy-of select="$savedTopic"/>
      </saxon:output>
    </xsl:when>

    <xsl:otherwise>
      <xsl:message terminate="yes">
        <xsl:text>Multiple result documents not implemented by </xsl:text>
        <xsl:value-of select="system-property('xsl:vendor')"/>
        <xsl:text>.</xsl:text>
      </xsl:message>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

</xsl:stylesheet>
