<!-- =========================================================================
 Copyright (c) 2017-2020 XMLmind Software. All rights reserved.
 
 Author: Hussein Shafie
 
 This file is part of the XMLmind W2X project.
 For conditions of distribution and use, see the accompanying legal.txt file.
========================================================================== -->

<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:h="http://www.w3.org/1999/xhtml"
  xmlns:m="http://www.w3.org/1998/Math/MathML"
  exclude-result-prefixes="h m">

<xsl:include href="common_params.xslt"/>
<xsl:include href="other_common.xslt"/>

<xsl:param name="processCellRotate" select="'yes'"/>
<xsl:include href="tgroup.xslt"/>

<xsl:output method="xml" encoding="UTF-8" indent="no"/>

<xsl:param name="hierarchy-name" select="'book'"/>

<xsl:variable name="hierarchies"
              select="document('docbook.hierarchy')/hierarchies"/>

<xsl:param name="hierarchyName">
  <xsl:choose>
    <xsl:when test="$hierarchies/hierarchy[@name = $hierarchy-name]">
      <xsl:value-of select="$hierarchy-name"/>
    </xsl:when>
    <xsl:otherwise>book</xsl:otherwise>
  </xsl:choose>
</xsl:param>

<xsl:variable name="hierarchy"
              select="$hierarchies/hierarchy[@name = $hierarchyName]"/>

<xsl:param name="docbook-version">
  <xsl:choose>
    <xsl:when test="$hierarchyName = 'topic'">5.1</xsl:when>
    <xsl:otherwise>4.5</xsl:otherwise>
  </xsl:choose>
</xsl:param>
<!--
    Syntax is:
    base_version-(subset|extension|variant) (name[-version])+

    Example:
    5.0-extension acme-1.0
-->
<xsl:param name="docbookVersion">
  <xsl:choose>
    <xsl:when test="contains($docbook-version, '-')">
      <xsl:value-of select="number(substring-before($docbook-version, '-'))"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="number($docbook-version)"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:param>

<xsl:param name="cals-tables" select="'no'"/>

<xsl:param name="media-alt" select="'no'"/>

<xsl:param name="pre-element-name" select="'literallayout'"/>

<xsl:template match="/">
  <xsl:if test="$docbookVersion &lt; 5">
    <xsl:text
      disable-output-escaping="yes">&lt;!DOCTYPE </xsl:text>
    <xsl:value-of select="$hierarchy/@root"/>
    <xsl:text
      disable-output-escaping="yes"> PUBLIC "-//OASIS//DTD DocBook XML V</xsl:text>
    <xsl:value-of select="$docbook-version"/>
    <xsl:text
      disable-output-escaping="yes">//EN" "http://www.oasis-open.org/docbook/xml/</xsl:text>
    <xsl:value-of select="$docbook-version"/>
    <xsl:text
      disable-output-escaping="yes">/docbookx.dtd"&gt;</xsl:text>
  </xsl:if>

  <xsl:apply-templates/>
</xsl:template>

<!-- html ================================================================ -->

<xsl:template match="h:html">
  <xsl:element name="{$hierarchy/@root}">
    <xsl:if test="$docbookVersion &gt;= 5">
      <xsl:attribute name="version">
        <xsl:value-of select="$docbook-version"/> <!-- '5.0', not just '5' -->
      </xsl:attribute>
    </xsl:if>

    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<!-- head, title, meta =================================================== -->

<xsl:template match="h:head">
  <xsl:if test="./h:meta[@name = 'description' or 
                         @name = 'author' or 
                         starts-with(@name, 'dc.') or
                         starts-with(@name, 'dcterms.')] or
                (./h:title and $hierarchy/@title-after-info != 'true')">
    <xsl:element name="{$hierarchy/@info}">
      <xsl:if test="$hierarchy/@title-after-info != 'true'">
        <xsl:apply-templates select="./h:title"/>
      </xsl:if>
      <xsl:apply-templates select="./h:meta"/>
    </xsl:element>
  </xsl:if>

  <xsl:if test="$hierarchy/@title-after-info = 'true'">
    <xsl:apply-templates select="./h:title"/>
  </xsl:if>
</xsl:template>

<xsl:template match="h:title |
                     h:div[starts-with(@class, 'role-section')]/h:h1 |
                     h:div[starts-with(@class, 'role-section')]/h:h2 |
                     h:div[starts-with(@class, 'role-section')]/h:h3 |
                     h:div[starts-with(@class, 'role-section')]/h:h4 |
                     h:div[starts-with(@class, 'role-section')]/h:h5 |
                     h:div[starts-with(@class, 'role-section')]/h:h6 |
                     h:p[@class = 'role-figcaption'] |
                     h:p[@class = 'role-equation-caption'] |
                     h:caption">
  <!-- h:caption processed here only when mode!=html-tables. -->
  <title>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </title>
</xsl:template>

<xsl:template match="h:meta">
  <xsl:choose>
    <xsl:when test="@name = 'description'">
      <abstract>
        <para><xsl:value-of select="@content"/></para>
      </abstract>
    </xsl:when>

    <xsl:when test="@name = 'author'">
      <author>
        <personname>
          <xsl:choose>
            <xsl:when test="$docbookVersion &lt; 5">
              <!-- DocBook 4 personname may not contain text. -->
              <othername><xsl:value-of select="@content"/></othername>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="@content"/>
            </xsl:otherwise>
          </xsl:choose>
        </personname>
      </author>
    </xsl:when>

    <xsl:when test="@name = 'dcterms.publisher'">
      <publishername>
        <xsl:value-of select="@content"/>
      </publishername>
    </xsl:when>

    <xsl:when test="@name = 'dcterms.created' or
                    @name = 'dcterms.modified'">
      <date remap="{@name}">
        <xsl:value-of select="@content"/>
      </date>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- body ================================================================ -->

<xsl:template match="h:body">
  <xsl:if test="$pasteFromWordOutput = 'yes'">
    <xsl:processing-instruction name="pfw-begin-body"/>
  </xsl:if>

  <xsl:variable name="boundary" 
                select="./*[self::h:div and 
                            (starts-with(@class, 'role-section') or
                             @class = 'role-footnotes' or
                             @class = 'role-footnote' or
                             @class = 'role-endnotes' or
                             @class = 'role-endnote')][1]"/>
  <xsl:choose>
    <xsl:when test="$boundary">
      <xsl:if test="$boundary/preceding-sibling::*">
        <xsl:choose>
          <xsl:when test="$hierarchy/@intro != ''">
            <xsl:element name="{$hierarchy/@intro}">
              <title></title>
              <xsl:apply-templates select="$boundary/preceding-sibling::*"/>
            </xsl:element>
          </xsl:when>
          <xsl:otherwise>
            <xsl:apply-templates select="$boundary/preceding-sibling::*"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:if>

      <xsl:apply-templates
          select="./*[self::h:div and starts-with(@class, 'role-section')]"/>
    </xsl:when>

    <xsl:otherwise>
      <!-- No boundary. Not even a footnote. -->
      <xsl:choose>
        <xsl:when test="$hierarchy/@intro != ''">
          <xsl:element name="{$hierarchy/@intro}">
            <title></title>
            <xsl:apply-templates/>
          </xsl:element>
        </xsl:when>
        <xsl:otherwise>
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>

  <xsl:if test="$pasteFromWordOutput = 'yes'">
    <xsl:processing-instruction name="pfw-end-body"/>
  </xsl:if>
</xsl:template>

<xsl:template match="h:div[starts-with(@class, 'role-section')]">
  <xsl:variable name="sectionType">
    <xsl:choose>
      <xsl:when test="@class = 'role-section1'">
        <xsl:value-of select="$hierarchy/@section1"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section2'">
        <xsl:value-of select="$hierarchy/@section2"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section3'">
        <xsl:value-of select="$hierarchy/@section3"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section4'">
        <xsl:value-of select="$hierarchy/@section4"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section5'">
        <xsl:value-of select="$hierarchy/@section5"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section6'">
        <xsl:value-of select="$hierarchy/@section6"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section7'">
        <xsl:value-of select="$hierarchy/@section7"/>
      </xsl:when>
      <xsl:when test="@class = 'role-section8'">
        <xsl:value-of select="$hierarchy/@section8"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$hierarchy/@section9"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:element name="{$sectionType}">
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:apply-templates select="./*[1]"/> <!-- h1, h2, ... or h6 -->

    <xsl:variable name="children" select="./*[position() >= 2]"/>
    <xsl:choose>
      <xsl:when test="$children">
        <xsl:apply-templates select="$children"/>
      </xsl:when>
      <xsl:otherwise><para/></xsl:otherwise>
    </xsl:choose>
  </xsl:element>
</xsl:template>

<!-- body elements ======================================================= -->

<!-- Blocks ========================= -->

<xsl:template match="h:center |
                     h:div[@class != 'role-figure' and
                           @class != 'role-equation' and
                           @class != 'role-bibliography' and
                           not(starts-with(@class, 'role-section')) and
                           @class != 'role-footnotes' and 
                           @class != 'role-footnote' and 
                           @class != 'role-endnotes' and 
                           @class != 'role-endnote']">
  <xsl:choose>
    <xsl:when test="@id != '' or @lang != ''">
      <!-- DocBook has no equivalent of a div. -->
      <blockquote remap="{local-name()}">
        <xsl:call-template name="processCommonAttributes"/>
        <xsl:apply-templates/>
      </blockquote>
    </xsl:when>

    <xsl:otherwise>
      <!-- Unwrap. -->
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:div[@class = 'role-figure']">
  <xsl:choose>
    <xsl:when test="$docbookVersion &lt; 5 and 
                    (parent::h:td or parent::h:th)">
      <!-- Keep DocBook 4 validator quiet. -->
      <para>
        <xsl:call-template name="processFigure"/>
      </para>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="processFigure"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processFigure">
  <xsl:variable name="figureType">
    <xsl:choose>
      <xsl:when test="./h:p[@class = 'role-figcaption']">figure</xsl:when>
      <xsl:otherwise>informalfigure</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:element name="{$figureType}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:div[@class = 'role-equation']">
  <xsl:choose>
    <xsl:when test="$docbookVersion &lt; 5 and 
                    (parent::h:td or parent::h:th)">
      <!-- Keep DocBook 4 validator quiet. -->
      <para>
        <xsl:call-template name="processEquation"/>
      </para>
    </xsl:when>
    <xsl:otherwise>
      <xsl:call-template name="processEquation"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processEquation">
  <xsl:variable name="equationType">
    <xsl:choose>
      <xsl:when
        test="./h:p[@class = 'role-equation-caption']">equation</xsl:when>
      <xsl:otherwise>informalequation</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:element name="{$equationType}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:div[@class = 'role-bibliography']">
  <bibliolist>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </bibliolist>
</xsl:template>

<xsl:template match="h:p[@class = 'role-bibliomixed']">
  <bibliomixed>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:choose>
      <xsl:when test="$docbookVersion &lt; 5">
        <!-- DocBook 4 bibliomixed is not a real mixed. 
             For example, it cannot contain emphasis. -->
        <xsl:call-template name="simpleInlineContent"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates/>
      </xsl:otherwise>
    </xsl:choose>
  </bibliomixed>
</xsl:template>

<xsl:template match="h:p">
  <xsl:choose>
    <xsl:when test="@class = 'role-inline-wrapper'">
      <para>
        <!-- Inline wrapper: no attributes. -->

        <xsl:if test="parent::h:body and count(../*) = 1">
          <xsl:if test="$pasteFromWordOutput = 'yes'">
            <xsl:processing-instruction name="pfw-inline-wrapper"/>
          </xsl:if>
        </xsl:if>

        <xsl:apply-templates/>
      </para>
    </xsl:when>

    <xsl:otherwise>
      <xsl:choose>
        <xsl:when test="parent::h:div[@class = 'role-figure'] and ./h:img">
          <xsl:choose>
            <xsl:when test="count(./node()) = 1">
              <!-- Unwrap. Just keep the image. -->
              <mediaobject>
                <xsl:call-template name="processCommonAttributes"/>
                <xsl:for-each select="./h:img">
                  <xsl:call-template name="processAltAttribute"/>
                  <xsl:call-template name="createImageobject"/>
                </xsl:for-each>
              </mediaobject>
            </xsl:when>

            <xsl:otherwise>
              <!-- DocBook 4 figure may not contain a para. -->
              <xsl:choose>
                <xsl:when test="$docbookVersion &lt; 5">
                  <blockquote>
                    <para>
                      <xsl:call-template name="processCommonAttributes"/>
                      <xsl:apply-templates/>
                    </para>
                  </blockquote>
                </xsl:when>
                <xsl:otherwise>
                  <para>
                    <xsl:call-template name="processCommonAttributes"/>
                    <xsl:apply-templates/>
                  </para>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>

        <xsl:when test="./h:object[@type='application/mathml+xml']">
          <xsl:choose>
            <xsl:when test="count(./node()) = 1">
              <!-- Unwrap. Just keep the object. -->
              <xsl:choose>
                <xsl:when test="parent::h:div[@class = 'role-equation']">
                  <xsl:call-template name="processMathMLMediaobject"/>
                </xsl:when>

                <xsl:otherwise>
                  <xsl:choose>
                    <xsl:when test="$docbookVersion &lt; 5 and 
                                    (parent::h:td or parent::h:th)">
                      <!-- Keep DocBook 4 validator quiet. -->
                      <para>
                        <informalequation>
                          <xsl:call-template name="processMathMLMediaobject"/>
                        </informalequation>
                      </para>
                    </xsl:when>
                    <xsl:otherwise>
                      <informalequation>
                        <xsl:call-template name="processMathMLMediaobject"/>
                      </informalequation>
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:when>

            <xsl:otherwise>
              <!-- Generates invalid docbook 4 or 5 when the parent is a 
                   div class=role-equation because an equation may not 
                   contain a para. -->
              <para>
                <xsl:call-template name="processCommonAttributes"/>
                <xsl:apply-templates/>
              </para>
            </xsl:otherwise>
          </xsl:choose>

        </xsl:when>

        <xsl:when test="./m:math[@display='block']">
          <!-- We assume here that the p only contains a math block. -->
          <xsl:choose>
            <xsl:when test="parent::h:div[@class = 'role-equation']">
              <xsl:for-each select="./m:math[@display='block']">
                <xsl:apply-templates select="." mode="math"/>
              </xsl:for-each>
            </xsl:when>

            <xsl:otherwise>
              <informalequation>
                <xsl:call-template name="processCommonAttributes"/>

                <xsl:for-each select="./m:math[@display='block']">
                  <xsl:apply-templates select="." mode="math"/>
                </xsl:for-each>
              </informalequation>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>

        <xsl:otherwise>
          <para>
            <xsl:call-template name="processCommonAttributes"/>
            <xsl:apply-templates/>
          </para>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processMathMLMediaobject">
  <mediaobject>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:for-each select="./h:object">
      <xsl:call-template name="createMathMLImageobject"/>
    </xsl:for-each>
  </mediaobject>
</xsl:template>

<xsl:template match="h:address">
  <para>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </para>
</xsl:template>

<xsl:template match="h:h1 | h:h2 | h:h3 | h:h4 | h:h5 | h:h6 |
                     h:p[starts-with(@class, 'role-bridgehead')]">
  <xsl:choose>
    <xsl:when test="$docbookVersion &lt; 5 and 
                    (parent::h:td or parent::h:th)">
      <para>
        <xsl:call-template name="processCommonAttributes"/>
        <xsl:apply-templates/>
      </para>
    </xsl:when>

    <xsl:otherwise>
      <xsl:call-template name="processBridgehead"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processBridgehead">
  <bridgehead>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:variable name="level">
      <xsl:choose>
        <xsl:when test="self::h:p">
          <xsl:value-of select="substring-after(@class, 'role-bridgehead')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="substring-after(local-name(), 'h')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:choose>
      <xsl:when test="number($level) &lt;= 5">
        <xsl:attribute name="renderas">
          <xsl:value-of select="concat('sect', $level)"/>
        </xsl:attribute>
      </xsl:when>

      <xsl:otherwise>
        <xsl:attribute name="renderas">other</xsl:attribute>
        <xsl:if test="$docbookVersion >= 5">
          <xsl:attribute name="otherrenderas">
            <xsl:value-of select="concat('sect', $level)"/>
          </xsl:attribute>
        </xsl:if>
      </xsl:otherwise>
    </xsl:choose>

    <xsl:apply-templates/>
  </bridgehead>
</xsl:template>

<xsl:template match="h:blockquote">
  <blockquote>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </blockquote>
</xsl:template>

<xsl:template match="h:pre">
  <xsl:element name="{$pre-element-name}">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </xsl:element>
</xsl:template>

<!-- Lists ========================= -->

<xsl:template match="h:ul | h:menu | h:dir">
  <itemizedlist>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>
    <xsl:apply-templates/>
  </itemizedlist>
</xsl:template>

<xsl:template name="processCompactAttribute">
  <xsl:if test="@compact">
    <xsl:attribute name="spacing">compact</xsl:attribute>
  </xsl:if>
</xsl:template>

<xsl:template match="h:li">
  <listitem>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:choose>
      <xsl:when test="./node()">
        <xsl:apply-templates/>
      </xsl:when>
      <xsl:otherwise><para/></xsl:otherwise>
    </xsl:choose>
  </listitem>
</xsl:template>

<xsl:template match="h:ol">
  <orderedlist>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>
    
    <xsl:if test="@type">
      <xsl:attribute name="numeration">
        <xsl:choose>
          <xsl:when test="@type = 'a'">loweralpha</xsl:when>
          <xsl:when test="@type = 'A'">upperalpha</xsl:when>
          <xsl:when test="@type = 'i'">lowerroman</xsl:when>
          <xsl:when test="@type = 'I'">upperroman</xsl:when>
          <xsl:otherwise>arabic</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute>
    </xsl:if>

    <xsl:if test="contains(@style, '-list-inherit-numbering')">
      <xsl:attribute name="inheritnum">inherit</xsl:attribute>
    </xsl:if>

    <xsl:if test="@start">
      <xsl:variable name="start">
        <xsl:call-template name="olStart"/>
      </xsl:variable>

      <xsl:choose>
        <xsl:when test="preceding-sibling::h:ol and 
                        @start = $start">
          <xsl:attribute name="continuation">continues</xsl:attribute>
        </xsl:when>
        <xsl:when test="@start != 1 and $docbookVersion >= 5">
          <xsl:attribute name="startingnumber">
            <xsl:value-of select="@start"/>
          </xsl:attribute>
        </xsl:when>
      </xsl:choose>
    </xsl:if>

    <xsl:apply-templates/>
  </orderedlist>
</xsl:template>

<xsl:template match="h:dl">
  <!-- Works only if the dl contains sequences comprising
       one or more dt followed by one or more dd. That is, a real dl. -->

  <variablelist>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processCompactAttribute"/>

    <xsl:for-each select="./h:dt">
      <xsl:variable name="previous" select="preceding-sibling::*[1]"/>
      <xsl:if test="not($previous) or $previous/self::h:dd">
        <varlistentry>
          <xsl:call-template name="dtGroup">
            <xsl:with-param name="head" select="."/>
          </xsl:call-template>

          <listitem>
            <xsl:variable name="firstDD"
                          select="./following-sibling::h:dd[1]"/>
            <xsl:for-each select="$firstDD">
              <xsl:call-template name="processCommonAttributes"/>
            </xsl:for-each>

            <xsl:call-template name="ddGroup">
              <xsl:with-param name="head" select="$firstDD"/>
            </xsl:call-template>
          </listitem>
        </varlistentry>
      </xsl:if>
    </xsl:for-each>
  </variablelist>
</xsl:template>

<xsl:template name="dtGroup">
  <xsl:param name="head" select="''"/>

  <xsl:for-each select="$head">
    <term>
      <xsl:call-template name="processCommonAttributes"/>
      <xsl:apply-templates/>
    </term>
  </xsl:for-each>

  <xsl:variable name="next" select="$head/following-sibling::*[1]"/>
  <xsl:if test="$next/self::h:dt">
    <xsl:call-template name="dtGroup">
      <xsl:with-param name="head" select="$next"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<xsl:template name="ddGroup">
  <xsl:param name="head" select="''"/>

  <xsl:for-each select="$head">
    <!-- Assumes that it contains one or more p. Unwrap. -->
    <xsl:apply-templates/>
  </xsl:for-each>

  <xsl:variable name="next" select="$head/following-sibling::*[1]"/>
  <xsl:if test="$next/self::h:dd">
    <xsl:call-template name="ddGroup">
      <xsl:with-param name="head" select="$next"/>
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- Table ========================= -->

<xsl:template match="h:table">
  <xsl:variable name="tableType">
    <xsl:choose>
      <xsl:when test="./h:caption">table</xsl:when>
      <xsl:otherwise>informaltable</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <xsl:choose>
    <xsl:when test="$cals-tables = 'yes'">
      <xsl:choose>
        <xsl:when test="$docbookVersion &lt; 5 and 
                        (parent::h:td or parent::h:th)">
          <!-- Keep DocBook 4 validator quiet. -->
          <para>
            <xsl:call-template name="processCALSTable">
              <xsl:with-param name="tableType" select="$tableType"/>
            </xsl:call-template>
          </para>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="processCALSTable">
            <xsl:with-param name="tableType" select="$tableType"/>
          </xsl:call-template>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <xsl:otherwise>
      <xsl:call-template name="processHTMLTable">
        <xsl:with-param name="tableType" select="$tableType"/>
      </xsl:call-template>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processCALSTable">
  <xsl:param name="tableType" select="'table'"/>

  <xsl:element name="{$tableType}">
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:call-template name="processTableBorderAttribute"/>

    <xsl:if test="./h:caption">
      <xsl:apply-templates select="./h:caption"/>
    </xsl:if>

    <xsl:call-template name="processTgroup"/>
  </xsl:element>
</xsl:template>

<xsl:template name="processHTMLTable">
  <xsl:param name="tableType" select="'table'"/>

  <xsl:element name="{$tableType}">
    <xsl:apply-templates select="@*|node()" mode="html-tables"/>
  </xsl:element>
</xsl:template>

<xsl:template match="h:caption" mode="html-tables">
  <caption>
    <xsl:apply-templates select="@*" mode="html-tables"/>

    <xsl:choose>
      <!-- DocBook 4 table/caption has the same content model 
           as mediaobject/caption. -->
      <xsl:when test="$docbookVersion &lt; 5 and not(./h:p)">
        <para>
          <xsl:apply-templates select="node()"/>
        </para>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates select="node()"/>
      </xsl:otherwise>
    </xsl:choose>
  </caption>
</xsl:template>

<xsl:template match="h:tr" mode="html-tables">
  <xsl:element name="{local-name()}">
    <xsl:apply-templates select="@*|node()" mode="html-tables"/>

    <xsl:if test="count(./h:th | ./h:td) = 0">
      <!-- In some rare cases, a row may be empty. -->
      <td/>
    </xsl:if>
  </xsl:element>
</xsl:template>

<xsl:template match="h:th | h:td" mode="html-tables">
  <xsl:element name="{local-name()}">
    <xsl:apply-templates select="@*" mode="html-tables"/>
    <xsl:apply-templates select="node()"/>
  </xsl:element>
</xsl:template>

<xsl:template match="*" mode="html-tables">
  <xsl:element name="{local-name()}">
    <xsl:apply-templates select="@*|node()" mode="html-tables"/>
  </xsl:element>
</xsl:template>

<xsl:template match="@*" mode="html-tables">
  <xsl:variable name="attrName" select="local-name()"/>
  <xsl:choose>
    <xsl:when test="$attrName = 'id'">
      <xsl:attribute name="id">
        <xsl:value-of select="."/>
      </xsl:attribute>
    </xsl:when>

    <xsl:when test="$attrName = 'lang'">
      <xsl:attribute name="lang">
        <xsl:value-of select="."/>
      </xsl:attribute>
    </xsl:when>

    <xsl:when test="$attrName != 'nowrap' and 
                   $attrName != 'height' and
                   ($attrName != 'width' or 
                    parent::h:table or parent::h:colgroup or parent::h:col) and
                   ($attrName != 'bgcolor' or $docbookVersion &lt; 5)">
      <xsl:copy/>
    </xsl:when>
  </xsl:choose>
</xsl:template>

<!-- Links ========================= -->

<xsl:key name="footnotes"
         match="h:div[@class = 'role-footnote' or @class = 'role-endnote']"
         use="@id"/>

<xsl:template match="h:a">
  <xsl:choose>
    <xsl:when test="@href != ''">
      <xsl:choose>
        <xsl:when test="starts-with(@href, '#') and
                        (@class = 'role-footnote-ref' or 
                         @class = 'role-endnote-ref')">
          <xsl:call-template name="processFn"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="processXref"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:when>

    <xsl:otherwise>
      <!-- Not a link nor a footnote. -->
      <xsl:choose>
        <xsl:when test="@name != '' or @id != '' or @lang != ''">
          <xsl:variable name="anchorType">
            <xsl:choose>
              <xsl:when test="./node()">phrase</xsl:when>
              <xsl:otherwise>anchor</xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:element name="{$anchorType}">
            <xsl:call-template name="processCommonAttributes"/>

            <xsl:if test="@name != '' and not(@id)">
              <xsl:attribute name="id">
                <xsl:value-of select="@name"/>
              </xsl:attribute>
            </xsl:if>

            <xsl:apply-templates/>
          </xsl:element>
        </xsl:when>

        <xsl:otherwise>
          <!-- Unwrap. -->
          <xsl:apply-templates/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template name="processFn">
  <xsl:variable name="footnote"
                select="key('footnotes', substring-after(@href, '#'))"/>
  <xsl:if test="$footnote">
    <footnote>
      <xsl:for-each select="$footnote">
        <xsl:call-template name="processCommonAttributes"/>
      </xsl:for-each>

      <xsl:apply-templates select="$footnote/node()"/>
    </footnote>
  </xsl:if>
</xsl:template>

<xsl:template name="processXref">
  <xsl:choose>
    <xsl:when test="starts-with(@href, '#')">
      <xsl:variable name="xrefType">
        <xsl:choose>
          <xsl:when test="@class = 'role-xref'">xref</xsl:when>
          <xsl:otherwise>link</xsl:otherwise>
        </xsl:choose>
      </xsl:variable>
      <xsl:element name="{$xrefType}">
        <xsl:call-template name="processCommonAttributes"/>

        <xsl:attribute name="linkend">
          <xsl:value-of select="substring-after(@href, '#')"/>
        </xsl:attribute>

        <xsl:apply-templates/>
      </xsl:element>
    </xsl:when>

    <xsl:otherwise>
      <ulink>
        <xsl:call-template name="processCommonAttributes"/>

        <xsl:attribute name="url"
                       >
          <xsl:value-of select="@href"/>
        </xsl:attribute>

        <xsl:apply-templates/>
      </ulink>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- Image ========================= -->

<xsl:template match="h:img">
  <inlinemediaobject>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="processAltAttribute"/>
    <xsl:call-template name="createImageobject"/>
  </inlinemediaobject>
</xsl:template>

<xsl:template name="processAltAttribute">
  <xsl:if test="$media-alt = 'yes' and @alt != ''">
    <alt><xsl:value-of select="normalize-space(@alt)"/></alt>
  </xsl:if>
</xsl:template>

<xsl:template name="createImageobject">
  <imageobject>
    <imagedata>
      <xsl:attribute name="fileref">
        <xsl:value-of select="@src"/>
      </xsl:attribute>

      <!-- Explicit '%' or implicit 'px' -->
      <xsl:if test="@width != ''">
        <xsl:attribute name="contentwidth">
          <xsl:value-of select="@width"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@height != ''">
        <xsl:attribute name="contentdepth">
          <xsl:value-of select="@height"/>
        </xsl:attribute>
      </xsl:if>
    </imagedata>
  </imageobject>
</xsl:template>

<!-- MathML ========================== -->

<xsl:template match="h:object[@type='application/mathml+xml']">
  <inlinemediaobject>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="createMathMLImageobject"/>
  </inlinemediaobject>
</xsl:template>

<xsl:template name="createMathMLImageobject">
  <imageobject>
    <imagedata>
      <xsl:attribute name="fileref">
        <xsl:value-of select="@data"/>
      </xsl:attribute>
    </imagedata>
  </imageobject>
</xsl:template>

<xsl:template match="m:math">
  <inlineequation>
    <xsl:apply-templates select="." mode="math"/>
  </inlineequation>
</xsl:template>

<!-- The DocBook 4 MathML module only supports the 'mml:' prefix.
     RELAX NG based DocBook 5 does not care. -->

<xsl:template match="m:*" mode="math">
  <xsl:element name="{concat('mml:', local-name())}"
               xmlns:mml="http://www.w3.org/1998/Math/MathML">
    <xsl:apply-templates select="@*|node()" mode="math"/>
  </xsl:element>
</xsl:template>

<xsl:template match="@*|node()" mode="math">
  <xsl:copy><xsl:apply-templates select="@*|node()" mode="math"/></xsl:copy>
</xsl:template>

<!-- Inlines ========================= -->

<xsl:template match="h:dfn">
  <firstterm>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </firstterm>
</xsl:template>

<xsl:template match="h:abbr">
  <abbrev>
    <xsl:call-template name="abbrevContent"/>
  </abbrev>
</xsl:template>

<xsl:template name="abbrevContent">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:choose>
      <xsl:when test="$docbookVersion &gt;= 5">
        <!-- DocBook 5 abbrev and acronym may not contain emphasis. -->
        <xsl:call-template name="simpleInlineContent2"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="simpleInlineContent1"/>
      </xsl:otherwise>
    </xsl:choose>
</xsl:template>

<xsl:template name="simpleInlineContent1">
  <xsl:call-template name="simpleInlineContent">
    <xsl:with-param name="allowEmphasis" select="true()"/>
    <xsl:with-param name="allowSubSup" select="true()"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="simpleInlineContent2">
  <xsl:call-template name="simpleInlineContent">
    <xsl:with-param name="allowEmphasis" select="false()"/>
    <xsl:with-param name="allowSubSup" select="true()"/>
  </xsl:call-template>
</xsl:template>

<xsl:template name="simpleInlineContent">
  <xsl:param name="allowEmphasis" select="false()"/>
  <xsl:param name="allowSubSup" select="false()"/>

  <xsl:for-each select="node()">
    <xsl:choose>
      <xsl:when test="self::h:em or self::h:i or
                      self::h:strong or self::h:b or
                      self::h:u or
                      self::h:span[@class = 'role-mark'] or
                      self::h:s or self::h:strike">
        <!-- Translated to emphasis. -->
        <xsl:choose>
          <xsl:when test="$allowEmphasis">
            <xsl:apply-templates select="."/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="simpleInlineContent">
              <xsl:with-param name="allowEmphasis" select="$allowEmphasis"/>
              <xsl:with-param name="allowSubSup" select="$allowSubSup"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:when test="self::h:sub or self::h:sup">
        <xsl:choose>
          <xsl:when test="$allowSubSup">
            <xsl:apply-templates select="."/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="simpleInlineContent">
              <xsl:with-param name="allowEmphasis" select="$allowEmphasis"/>
              <xsl:with-param name="allowSubSup" select="$allowSubSup"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>

      <xsl:when test="self::h:dfn or
                      self::h:abbr or
                      self::h:acronym or
                      self::h:code or
                      self::h:var or
                      self::h:kbd or
                      self::h:samp or
                      self::h:tt or
                      self::h:cite or
                      self::h:q or
                      self::h:big or self::h:font or
                      self::h:span or self::h:small">
        <xsl:call-template name="simpleInlineContent">
          <xsl:with-param name="allowEmphasis" select="$allowEmphasis"/>
          <xsl:with-param name="allowSubSup" select="$allowSubSup"/>
        </xsl:call-template>
      </xsl:when>

      <xsl:otherwise>
        <xsl:apply-templates select="."/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>

<xsl:template match="h:acronym">
  <acronym>
    <xsl:call-template name="abbrevContent"/>
  </acronym>
</xsl:template>

<xsl:template match="h:code">
  <code>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="simpleInlineContent2"/>
  </code>
</xsl:template>

<xsl:template match="h:var">
  <varname>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="simpleInlineContent"/>
  </varname>
</xsl:template>

<xsl:template match="h:kbd">
  <userinput>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="simpleInlineContent2"/>
  </userinput>
</xsl:template>

<xsl:template match="h:samp">
  <computeroutput>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="simpleInlineContent2"/>
  </computeroutput>
</xsl:template>

<xsl:template match="h:em | h:i">
  <emphasis>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </emphasis>
</xsl:template>

<xsl:template match="h:strong | h:b">
  <emphasis role="bold">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </emphasis>
</xsl:template>

<xsl:template match="h:u">
  <emphasis role="underline">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </emphasis>
</xsl:template>

<xsl:template match="h:s | h:strike">
  <emphasis role="strikethrough">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </emphasis>
</xsl:template>

<xsl:template match="h:span[@class = 'role-mark']">
  <emphasis role="highlight">
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </emphasis>
</xsl:template>

<xsl:template match="h:tt">
  <literal>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:call-template name="simpleInlineContent2"/>
  </literal>
</xsl:template>

<xsl:template match="h:cite">
  <citetitle>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </citetitle>
</xsl:template>

<xsl:template match="h:q">
  <quote>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </quote>
</xsl:template>

<xsl:template match="h:sub">
  <subscript>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:choose>
      <xsl:when test="$docbookVersion &lt; 5">
        <xsl:call-template name="simpleInlineContent1"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="simpleInlineContent2"/>
      </xsl:otherwise>
    </xsl:choose>
  </subscript>
</xsl:template>

<xsl:template match="h:sup[.//h:a[@class = 'role-footnote-ref' or 
                                  @class = 'role-endnote-ref']]">
  <!-- Common case: unwrap sup as a footnote is not allowed 
       in a superscript. -->
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="h:sup">
  <superscript>
    <xsl:call-template name="processCommonAttributes"/>

    <xsl:choose>
      <xsl:when test="$docbookVersion &lt; 5">
        <xsl:call-template name="simpleInlineContent1"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="simpleInlineContent2"/>
      </xsl:otherwise>
    </xsl:choose>
  </superscript>
</xsl:template>

<xsl:template match="h:ruby">
  <phrase>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:value-of select="."/>
  </phrase>
</xsl:template>

<xsl:template match="h:hr |
                     h:br">
  <xsl:text> </xsl:text>
</xsl:template>

<xsl:template match="h:span[@class = 'role-citation']">
  <citation>
    <xsl:call-template name="processCommonAttributes"/>
    <xsl:apply-templates/>
  </citation>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-start']">
  <indexterm class="startofrange" 
             id="{concat('__RANGE', substring-after(@id, '__IDX_S_'))}">
    <xsl:apply-templates/>
  </indexterm>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-end']">
  <indexterm class="endofrange" 
             startref="{concat('__RANGE', substring-after(@id, '__IDX_E_'))}"/>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term']">
  <indexterm><xsl:apply-templates/></indexterm>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-1']">
  <primary>
    <xsl:if test="@title">
      <xsl:attribute name="sortas">
        <xsl:value-of select="@title"/>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates/>
  </primary>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-2']">
  <secondary>
    <xsl:if test="@title">
      <xsl:attribute name="sortas">
        <xsl:value-of select="@title"/>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates/>
  </secondary>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-3']">
  <tertiary>
    <xsl:if test="@title">
      <xsl:attribute name="sortas">
        <xsl:value-of select="@title"/>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates/>
  </tertiary>
</xsl:template>

<xsl:template match="h:span[@class = 'role-index-term-see']">
  <see><xsl:apply-templates/></see>
</xsl:template>

<xsl:template match="h:big |
                     h:font |
                     h:span |
                     h:small">
  <xsl:choose>
    <xsl:when test="@id != '' or @lang != ''">
      <phrase>
        <xsl:call-template name="processCommonAttributes"/>
        <xsl:apply-templates/>
      </phrase>
    </xsl:when>

    <xsl:otherwise>
      <!-- Unwrap. -->
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h:ins">
  <!-- Unwrap. -->
  <xsl:apply-templates/>
</xsl:template>

<xsl:template match="h:del"/>

<!-- Attributes ========================================================== -->

<xsl:template name="processCommonAttributes">
  <xsl:call-template name="processIdAttribute"/>
  <xsl:call-template name="processLangAttribute"/>
</xsl:template>

<xsl:template name="processIdAttribute">
  <xsl:if test="@id != ''">
    <xsl:attribute name="id">
      <xsl:value-of select="@id"/>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<xsl:template name="processLangAttribute">
  <xsl:if test="@lang != ''">
    <xsl:attribute name="lang">
      <xsl:value-of select="@lang"/>
    </xsl:attribute>
  </xsl:if>
</xsl:template>

<!-- Not translated ==========================================================
applet
area
base
basefont
bdo
button
fieldset
form
iframe
input
isindex
label
legend
link
map
noframes
noscript
object (in the general case)
optgroup
option
param
script
select
style
textarea
========================================================================== -->

<xsl:template match="*">
  <xsl:message>Internal error: XHTML element '<xsl:value-of select="local-name(.)"/>' has not been translated to DocBook. Please report a bug to xmleditor-support@xmlmind.com.</xsl:message>

  <xsl:element name="{local-name()}" namespace="http://www.w3.org/1999/xhtml">
    <xsl:apply-templates select="@*|node()" mode="asis"/>
  </xsl:element>
</xsl:template>

<xsl:template match="@*|node()" mode="asis">
  <xsl:copy><xsl:apply-templates select="@*|node()" mode="asis"/></xsl:copy>
</xsl:template>

</xsl:stylesheet>
